import 'dart:async';
import 'package:admin_desktop/app_constants.dart';
import 'package:admin_desktop/application/providers.dart';
import 'package:admin_desktop/presentation/components/dialogs/delete_dialog.dart';
import 'package:auto_route/auto_route.dart';
import 'package:auto_size_text/auto_size_text.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import 'package:proste_indexed_stack/proste_indexed_stack.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import 'package:admin_desktop/presentation/routes/app_router.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'widgets/booking/booking_statistics_page.dart';
import 'widgets/bottom_bar/widgets/custom_bottom_bar.dart';
import 'widgets/bottom_bar/widgets/custom_master_bottom_bar.dart';
import 'widgets/catalogue/catalogue_page.dart';
import 'widgets/comments/comments_page.dart';
import 'widgets/income/income_page.dart';
import 'widgets/language/languages_modal.dart';
import 'widgets/main_appbar.dart';
import 'widgets/orders_table/orders_table.dart';
import 'widgets/post_page.dart';
import 'widgets/profile/profile_page.dart';
import 'widgets/sale_history/sale_history.dart';
import 'widgets/schedule/schedule_page.dart';
import 'widgets/service_master/service_master_table_page.dart';

@RoutePage()
class MainPage extends ConsumerStatefulWidget {
  const MainPage({super.key});

  @override
  ConsumerState<MainPage> createState() => _MainPageState();
}

class _MainPageState extends ConsumerState<MainPage>
    with SingleTickerProviderStateMixin {
  final user = LocalStorage.getUser();

  late List<IndexedStackChild> masterList = [
    IndexedStackChild(child: const SchedulePage(), preload: true),
    IndexedStackChild(child: const BookingStatisticsPage()),
    IndexedStackChild(child: const ProfilePage()),
    IndexedStackChild(child: const ServiceMasterTablePage()),
  ];

  late List<IndexedStackChild> list = [
    IndexedStackChild(child: const PostPage(), preload: true),
    IndexedStackChild(child: const OrdersTablesPage()),
    IndexedStackChild(child: const SchedulePage()),
    IndexedStackChild(child: const CommentsPage()),
    IndexedStackChild(child: const SaleHistory()),
    IndexedStackChild(child: const InComePage()),
    IndexedStackChild(child: const BookingStatisticsPage()),
    IndexedStackChild(child: const CataloguePage()),
    IndexedStackChild(child: const ProfilePage()),
  ];

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_) {
      ref.read(mainProvider.notifier)
        ..changeIndex(0)
        ..fetchMyShop()
        ..fetchUserDetail(onSuccess: () {
          ref.read(mainProvider.notifier)
            ..fetchProducts(
              isRefresh: true,
              checkYourNetwork: () {
                AppHelpers.showSnackBar(
                  context,
                  AppHelpers.getTranslation(TrKeys.checkYourNetworkConnection),
                );
              },
            )
            ..fetchCategories(
              context: context,
              checkYourNetwork: () {
                AppHelpers.showSnackBar(
                  context,
                  AppHelpers.getTranslation(TrKeys.checkYourNetworkConnection),
                );
              },
            );
          ref.read(languagesProvider.notifier).checkLanguage();
        });

      ref.read(rightSideProvider.notifier).fetchUsers(
        checkYourNetwork: () {
          AppHelpers.showSnackBar(
            context,
            AppHelpers.getTranslation(TrKeys.checkYourNetworkConnection),
          );
        },
      );
      if (mounted) {
        Timer.periodic(
          AppConstants.refreshTime,
          (s) {
            ref.read(notificationProvider.notifier).fetchCount(context);
          },
        );
      }
    });
  }

  @override
  Widget build(BuildContext context) {
    ref.listen(
      languagesProvider,
      (previous, next) {
        if (next.isSelectLanguage &&
            next.isSelectLanguage != previous?.isSelectLanguage) {
          AppHelpers.showAlertDialog(
            context: context,
            child: SizedBox(
              height: MediaQuery.sizeOf(context).height / 2,
              width: MediaQuery.sizeOf(context).width / 4,
              child: LanguagesModal(
                afterUpdate: () {
                  context.maybePop();
                },
              ),
            ),
          );
        }
      },
    );

    return SafeArea(
      child: KeyboardDismisser(
        child: Consumer(
          builder: (BuildContext context, WidgetRef ref, Widget? child) {
            final state = ref.watch(mainProvider);
            return CustomScaffold(
              extendBody: true,
              backgroundColor: Style.mainBack,
              body: (c) => Directionality(
                textDirection: LocalStorage.getLangLtr()
                    ? TextDirection.ltr
                    : TextDirection.rtl,
                child: Row(
                  children: [
                    bottomLeftNavigationBar(state),
                    Expanded(
                      child: Column(
                        children: [
                          MainAppbar(),
                          Expanded(
                            child: ProsteIndexedStack(
                              index: state.selectIndex,
                              children: AppHelpers.getUserRole == TrKeys.master
                                  ? masterList
                                  : list,
                            ),
                          ),
                        ],
                      ),
                    ),
                  ],
                ),
              ),
            );
          },
        ),
      ),
    );
  }

  Container bottomLeftNavigationBar(MainState state) {
    return Container(
      height: double.infinity,
      width: state.active ? 150 : 60,
      color: Style.black,
      //  duration: Duration(milliseconds: 300),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.center,
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          24.verticalSpace,
          Row(
            children: [
              8.horizontalSpace,
              if (state.active)
                Expanded(
                  child: AutoSizeText(
                    AppHelpers.getAppName(),
                    style: Style.interSemi(color: Style.white),
                    maxLines: 1,
                    minFontSize: 14,
                  ),
                ),
              CircleIconButton(
                  size: 30,
                  backgroundColor: Style.white,
                  iconData: state.active
                      ? Remix.arrow_left_line
                      : Remix.arrow_right_line,
                  iconColor: Style.black,
                  onTap: () => ref.read(mainProvider.notifier).changeActive()),
              5.horizontalSpace,
            ],
          ),
          28.verticalSpace,
          AppHelpers.getUserRole == TrKeys.master
              ? CustomMasterBottomBar(
                  state: state,
                  notifier: ref.read(mainProvider.notifier),
                )
              : CustomBottomBar(
                  state: state,
                  notifier: ref.read(mainProvider.notifier),
                ),
          const Spacer(),
          Row(
            mainAxisAlignment: state.active
                ? MainAxisAlignment.start
                : MainAxisAlignment.center,
            children: [
              6.horizontalSpace,
              IconButton(
                  onPressed: () {
                    AppHelpers.showAlertDialog(
                        context: context,
                        child: DeleteDialog(success: () {
                          context.replaceRoute(const LoginRoute());
                          ref.read(newOrdersProvider.notifier).stopTimer();
                          ref.read(acceptedOrdersProvider.notifier).stopTimer();
                          ref.read(readyOrdersProvider.notifier).stopTimer();
                          ref.read(onAWayOrdersProvider.notifier).stopTimer();
                          ref
                              .read(deliveredOrdersProvider.notifier)
                              .stopTimer();
                          ref.read(canceledOrdersProvider.notifier).stopTimer();
                          LocalStorage.clearStore();
                        }));
                  },
                  icon: const Icon(
                    Remix.logout_circle_line,
                    color: Style.white,
                  )),
              5.horizontalSpace,
              if (state.active)
                Text(
                  AppHelpers.getTranslation(TrKeys.logout),
                  style: Style.interMedium(
                    size: 12,
                    color: Style.white,
                  ),
                ),
            ],
          ),
          32.verticalSpace,
        ],
      ),
    );
  }
}
